from PIL import Image, ImageDraw, ImageFont
import os

# === CONFIGURACIÓN ===
CARPETA_ORIGEN = "."           # Carpeta raíz con las imágenes
CARPETA_SALIDA = "../Con Nombre"    # Carpeta raíz de salida
FUENTE = "arial.ttf"           # Fuente
POS_Y_PCT = 0.92               # Altura del texto (porcentaje desde arriba)
ANCHO_TEXTO_PCT = 0.6          # Porcentaje del ancho máximo que puede ocupar el texto
MAX_FUENTE_PCT = 0.15          # Tamaño máximo de fuente relativo a la altura de la imagen

# === FUNCIONES ===

def ajustar_fuente(draw, texto, fuente_path, ancho_max, alto_img):
    """Calcula el tamaño máximo de fuente que entra en el ancho/alto permitido sin pasar del límite relativo."""
    tamaño = 1
    fuente = ImageFont.truetype(fuente_path, tamaño)
    max_tamano = int(alto_img * MAX_FUENTE_PCT)
    while True:
        bbox = draw.textbbox((0, 0), texto, font=fuente)
        w, h = bbox[2] - bbox[0], bbox[3] - bbox[1]
        if w > ancho_max or h > alto_img * 0.2 or tamaño >= max_tamano:
            break
        tamaño += 1
        fuente = ImageFont.truetype(fuente_path, tamaño)
    return ImageFont.truetype(fuente_path, max(1, tamaño - 1))


def procesar_imagen(ruta_entrada, ruta_salida, nombre_archivo):
    """Procesa una imagen individual, agrega texto y la guarda."""
    img = Image.open(ruta_entrada).convert("RGBA")
    draw = ImageDraw.Draw(img)
    nombre = os.path.splitext(nombre_archivo)[0]

    ancho_img, alto_img = img.size
    ancho_max_texto = ancho_img * ANCHO_TEXTO_PCT
    fuente = ajustar_fuente(draw, nombre, FUENTE, ancho_max_texto, alto_img)

    bbox = draw.textbbox((0, 0), nombre, font=fuente)
    w, h = bbox[2] - bbox[0], bbox[3] - bbox[1]

    # Centramos horizontalmente y alineamos por abajo
    x = (ancho_img - w) / 2
    y = alto_img * POS_Y_PCT - h

    # Contorno proporcional (5 % del tamaño de fuente)
    outline = int(max(1, fuente.size * 0.07))
    for dx in range(-outline, outline + 1):
        for dy in range(-outline, outline + 1):
            if dx != 0 or dy != 0:
                draw.text((x + dx, y + dy), nombre, font=fuente, fill="black")

    # Texto principal
    draw.text((x, y), nombre, font=fuente, fill="white")

    # Crear carpeta de salida si no existe
    os.makedirs(os.path.dirname(ruta_salida), exist_ok=True)

    # Guardar según formato
    if nombre_archivo.lower().endswith(".png"):
        img.save(ruta_salida, "PNG")
    else:
        img.convert("RGB").save(ruta_salida, "JPEG")

    print(f"✔ {ruta_entrada} → {ruta_salida} ({fuente.size}px)")


# === PROCESO PRINCIPAL ===

def recorrer_carpeta(origen, salida):
    """Recorre todas las subcarpetas y procesa imágenes."""
    total = 0
    for carpeta_actual, subcarpetas, archivos in os.walk(origen):
        for archivo in archivos:
            if archivo.lower().endswith((".png", ".jpg", ".jpeg")):
                ruta_entrada = os.path.join(carpeta_actual, archivo)
                rel_path = os.path.relpath(carpeta_actual, origen)
                ruta_salida = os.path.join(salida, rel_path, archivo)
                procesar_imagen(ruta_entrada, ruta_salida, archivo)
                total += 1
    print(f"\n✅ Se procesaron {total} imágenes en total.\n")


if __name__ == "__main__":
    recorrer_carpeta(CARPETA_ORIGEN, CARPETA_SALIDA)
